;; gnome.jl -- minimal GNOME compliance
;; $Id: gnome.jl,v 1.48 2000/03/07 18:11:22 john Exp $

;; Copyright (C) 1999 John Harper <john@dcs.warwick.ac.uk>

;; This file is part of sawmill.

;; sawmill is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; sawmill is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with sawmill; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

(require 'workspace)
(require 'viewport)
(require 'maximize)
(provide 'gnome)

(defconst WIN_STATE_STICKY 1)
(defconst WIN_STATE_MAXIMIZED_VERT 4)
(defconst WIN_STATE_MAXIMIZED_HORIZ 8)
(defconst WIN_STATE_HIDDEN 16)
(defconst WIN_STATE_SHADED 32)

(defconst WIN_LAYER_NORMAL 4)

(defconst WIN_HINTS_SKIP_FOCUS 1)
(defconst WIN_HINTS_SKIP_WINLIST 2)
(defconst WIN_HINTS_SKIP_TASKLIST 4)
(defconst WIN_HINTS_FOCUS_ON_CLICK 16)

(defvar gnome-window-id nil)

(defvar gnome-supported-protocols [_WIN_CLIENT_LIST _WIN_WORKSPACE
				   _WIN_WORKSPACE_COUNT _WIN_STATE
				   _WIN_LAYER])

;; this is needed since the gnome tasklist applet doesn't honour
;; the _WIN_HIDDEN property (?)
(defvar gnome-ignored-windows-in-client-list t)

(defun gnome-set-client-list ()
  (let
      (clients vec)
    (mapc (lambda (w)
	    (when (and (windowp w) (window-mapped-p w)
		       (or gnome-ignored-windows-in-client-list
			   (not (window-get w 'ignored))))
	      (setq clients (cons (window-id w) clients))))
	  (managed-windows))
    (setq vec (apply vector clients))
    (set-x-property 'root '_WIN_CLIENT_LIST vec 'CARDINAL 32)))

(defvar gnome-current-workspace nil)
(defvar gnome-current-workspace-count 0)
(defvar gnome-current-workspace-names nil)
(defvar gnome-current-area nil)
(defvar gnome-current-area-count nil)

(defun gnome-set-workspace ()
  (let*
      ((limits (workspace-limits))
       (port (screen-viewport))
       (port-size (cons viewport-columns viewport-rows))
       (total-workspaces (1+ (- (cdr limits) (car limits)))))
    ;; apparently some pagers don't like it if we place windows
    ;; on (temporarily) non-existent workspaces
    (when (< gnome-current-workspace-count total-workspaces)
      (setq gnome-current-workspace-count total-workspaces)
      (set-x-property 'root '_WIN_WORKSPACE_COUNT
		      (vector gnome-current-workspace-count) 'CARDINAL 32))
    (mapc (lambda (w)
	    (let
		;; XXX the gnome-wm standard sucks..
		((space (and (not (window-get w 'sticky))
			     (window-get w 'swapped-in)))
		 (w-port (and (not (window-get w 'viewport-sticky))
			      (window-viewport w))))
	      (if space
		  (set-x-property w '_WIN_WORKSPACE
				  (vector (- space (car limits))) 'CARDINAL 32)
		(delete-x-property w '_WIN_WORKSPACE))
	      (if w-port
		  (set-x-property w '_WIN_AREA
				  (vector (car w-port) (cdr w-port))
				  'CARDINAL 32)
		(delete-x-property w '_WIN_AREA))))
	  (managed-windows))
    (unless (equal gnome-current-workspace (- current-workspace (car limits)))
      (setq gnome-current-workspace (- current-workspace (car limits)))
      (set-x-property 'root '_WIN_WORKSPACE
		      (vector gnome-current-workspace) 'CARDINAL 32))
    (when (> gnome-current-workspace-count total-workspaces)
      (setq gnome-current-workspace-count total-workspaces)
      (set-x-property 'root '_WIN_WORKSPACE_COUNT
		      (vector gnome-current-workspace-count) 'CARDINAL 32))
    (unless (equal gnome-current-workspace-names workspace-names)
      (setq gnome-current-workspace-names workspace-names)
      (set-x-text-property 'root '_WIN_WORKSPACE_NAMES
			   (apply vector workspace-names)))
    (unless (equal gnome-current-area port)
      (setq gnome-current-area port)
      (set-x-property 'root '_WIN_AREA (vector (car port) (cdr port))
		      'CARDINAL 32))
    (unless (equal gnome-current-area-count port-size)
      (setq gnome-current-area-count port-size)
      (set-x-property 'root '_WIN_AREA_COUNT (vector (car port-size)
						     (cdr port-size))
		      'CARDINAL 32))))

(defun gnome-set-client-state (w)
  (let
      ((state 0))
    (when (window-get w 'sticky)
      (setq state (logior state WIN_STATE_STICKY)))
    (when (window-get w 'shaded)
      (setq state (logior state WIN_STATE_SHADED)))
    (when (window-maximized-vertically-p w)
      (setq state (logior state WIN_STATE_MAXIMIZED_VERT)))
    (when (window-maximized-horizontally-p w)
      (setq state (logior state WIN_STATE_MAXIMIZED_HORIZ)))
    (when (window-get w 'ignored)
      (setq state (logior state WIN_STATE_HIDDEN)))
    (set-x-property w '_WIN_STATE (vector state) 'CARDINAL 32)
    (when (window-get w 'depth)
      (set-x-property w '_WIN_LAYER
		      (vector (+ (window-get w 'depth) WIN_LAYER_NORMAL))
		      'CARDINAL 32))))

(defun gnome-honour-client-state (w)
  (let
      ((class (get-x-text-property w 'WM_CLASS)))
    (when (and class (>= (length class) 2))
      (cond ((and (string= (aref class 1) "Panel")
		  (string= (aref class 0) "panel"))
	     ;; XXX I don't think the GNOME hints specify these things
	     (window-put w 'focus-click-through t)
	     (window-put w 'avoid t))
	    ((string= (aref class 1) "gmc-desktop-icon")
	     (window-put w 'never-focus t)))))
  (let
      ((state (get-x-property w '_WIN_STATE))
       (hints (get-x-property w '_WIN_HINTS))
       (layer (get-x-property w '_WIN_LAYER))
       (space (get-x-property w '_WIN_WORKSPACE))
       bits)
    (when (eq (car state) 'CARDINAL)
      (setq bits (aref (nth 2 state) 0))
      (unless (zerop (logand bits WIN_STATE_STICKY))
	(window-put w 'sticky t)
	(window-put w 'sticky-viewport t))
      (unless (zerop (logand bits WIN_STATE_SHADED))
	(window-put w 'shaded t))
;;; XXX this doesn't work since the frame hasn't been created yet..
;      (unless (zerop (logand bits WIN_STATE_MAXIMIZED_VERT))
;	(unless (window-maximized-vertically-p w)
;	  (maximize-window-vertically w)))
;      (unless (zerop (logand bits WIN_STATE_MAXIMIZED_HORIZ))
;	(unless (window-maximized-horizontally-p w)
;	  (maximize-window-horizontally w)))
      )
    (when (eq (car hints) 'CARDINAL)
      (setq bits (aref (nth 2 hints) 0))
      (unless (zerop (logand bits WIN_HINTS_SKIP_FOCUS))
	(window-put w 'ignored t)))
    (when layer
      (setq layer (aref (nth 2 layer) 0))
      (set-window-depth w (- layer WIN_LAYER_NORMAL)))
    (when (and space (not (window-workspaces w)))
      (window-add-to-workspace w (aref (nth 2 space) 0)))))

(defun gnome-client-message-handler (w type data)
  (cond ((eq type '_WIN_WORKSPACE)
	 (let
	     ((limits (workspace-limits)))
	   (select-workspace (+ (aref data 0) (car limits)))
	   t))
	((eq type '_WIN_AREA)
	 (set-screen-viewport (aref data 0) (aref data 1)))
	((and (eq type '_WIN_STATE) (windowp w))
	 (let
	     ((mask (aref data 0))
	      (values (aref data 1))
	      tem)
	   (unless (zerop (logand mask WIN_STATE_STICKY))
	     (if (zerop (logand values WIN_STATE_STICKY))
		 (make-window-unsticky w)
	       (make-window-sticky w)))
	   (unless (zerop (logand mask WIN_STATE_SHADED))
	     (setq tem (window-get w 'shaded))
	     (if (zerop (logand values WIN_STATE_SHADED))
		 (unshade-window w)
	       (shade-window w)))
	   (unless (zerop (logand mask WIN_STATE_MAXIMIZED_VERT))
	     (setq tem (window-maximized-vertically-p w))
	     (if (or (and (not tem) (not (zerop (logand values WIN_STATE_MAXIMIZED_VERT))))
		     (and tem (zerop (logand values WIN_STATE_MAXIMIZED_VERT))))
		 (maximize-window-vertically-toggle w)))
	   (unless (zerop (logand mask WIN_STATE_MAXIMIZED_HORIZ))
	     (setq tem (window-maximized-horizontally-p w))
	     (if (or (and (not tem) (not (zerop (logand values WIN_STATE_MAXIMIZED_HORIZ))))
		     (and tem (zerop (logand values WIN_STATE_MAXIMIZED_HORIZ))))
		 (maximize-window-horizontally-toggle w))))
	 t)
	((and (eq type '_WIN_LAYER) (windowp w))
	 (set-window-depth w (- (aref data 0) WIN_LAYER_NORMAL))
	 t)))

(defun gnome-event-proxyer ()
  (when (and (current-event) (eq (current-event-window) 'root))
    (let
	((event (event-name (current-event))))
      ;; only proxy Click1 or Off events, and only if we don't have
      ;; a binding for an event that may follow in the same grab
      (cond ((and (string-match "^(.*)-Click1$" event)
		  (let
		      ((mirror (lookup-event (expand-last-match "\\1-Off"))))
		    (not (or (search-keymap mirror global-keymap)
			     (search-keymap mirror root-window-keymap)))))
	     ;; send with SubstructureNotifyMask
	     (proxy-current-event gnome-window-id (lsh 1 19))
	     t)
	    ((and (string-match "^(.*)-Off$" event)
		  (let
		      ((mirrors
			(mapcar (lambda (x)
				  (lookup-event
				   (concat (expand-last-match "\\1-") x)))
				'("Click1" "Click2" "Click3" "Move"))))
		    (catch 'out
		      (mapc (lambda (ev)
			      (when (or (search-keymap ev global-keymap)
					(search-keymap ev root-window-keymap))
				(throw 'out nil)))
			    mirrors)
		      t)))
	     ;; send with SubstructureNotifyMask
	     (proxy-current-event gnome-window-id (lsh 1 19))
	     t)))))


;; initialisation

(defun gnome-init ()
  (setq gnome-window-id (create-window 'root -200 -200 5 5))

  (set-x-property 'root '_WIN_SUPPORTING_WM_CHECK
		  (vector gnome-window-id) 'CARDINAL 32)
  (set-x-property gnome-window-id '_WIN_SUPPORTING_WM_CHECK
		  (vector gnome-window-id) 'CARDINAL 32)

  (set-x-property 'root '_WIN_DESKTOP_BUTTON_PROXY
		  (vector gnome-window-id) 'CARDINAL 32)
  (set-x-property gnome-window-id '_WIN_DESKTOP_BUTTON_PROXY
		  (vector gnome-window-id) 'CARDINAL 32)

  (set-x-property 'root '_WIN_PROTOCOLS
		  gnome-supported-protocols 'ATOM 32)

  (let
      ((port (screen-viewport)))
    (set-x-property 'root '_WIN_AREA
		    (vector (car port) (cdr port)) 'CARDINAL 32)
    (set-x-property 'root '_WIN_AREA_COUNT
		    (vector viewport-columns viewport-rows) 'CARDINAL 32)

    ;; XXX I'm using this property to tell desk-guide to move
    ;; XXX the current area on all desktops at once
    ;; XXX This is totally non-standard and may change..
    (set-x-property 'root '_WIN_UNIFIED_AREA (vector 1) 'CARDINAL 32))

  (delete-x-property 'root '_WIN_WORKSPACE_NAMES)

  (add-hook 'workspace-state-change-hook gnome-set-workspace)
  (add-hook 'viewport-resized-hook gnome-set-workspace)
  (add-hook 'viewport-moved-hook gnome-set-workspace)

  (add-hook 'add-window-hook gnome-set-client-list)
  (add-hook 'destroy-notify-hook gnome-set-client-list)
  (add-hook 'map-notify-hook gnome-set-client-list)
  (add-hook 'unmap-notify-hook gnome-set-client-list)
  (add-hook 'workspace-state-change-hook gnome-set-client-list)

  (add-hook 'before-add-window-hook gnome-honour-client-state)
  (add-hook 'add-window-hook gnome-set-client-state)
  (call-after-state-changed '(sticky shaded maximized ignored stacking)
			    gnome-set-client-state)

  (add-hook 'client-message-hook gnome-client-message-handler)
  (add-hook 'unbound-key-hook gnome-event-proxyer)
  (add-hook 'before-exit-hook gnome-exit))

(defun gnome-exit ()
  (destroy-window gnome-window-id)
  (delete-x-property 'root '_WIN_SUPPORTING_WM_CHECK)
  (delete-x-property 'root '_WIN_PROTOCOLS)
  (delete-x-property 'root '_WIN_AREA)
  (delete-x-property 'root '_WIN_AREA_COUNT)
  (delete-x-property 'root '_WIN_UNIFIED_AREA))

(unless (or gnome-window-id batch-mode)
  (gnome-init)

  ;; arrange for gnome-match to be loaded when necessary..
  (if (featurep 'match-window)
      (require 'gnome-match)
    (eval-after-load "match-window" '(require 'gnome-match))))
