/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.wali;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.regex.Pattern;
import org.apache.nifi.wali.BlockingQueuePool;
import org.apache.nifi.wali.ByteArrayDataOutputStream;
import org.apache.nifi.wali.HashMapSnapshot;
import org.apache.nifi.wali.JournalRecovery;
import org.apache.nifi.wali.JournalSummary;
import org.apache.nifi.wali.LengthDelimitedJournal;
import org.apache.nifi.wali.ObjectPool;
import org.apache.nifi.wali.RecordLookup;
import org.apache.nifi.wali.SnapshotCapture;
import org.apache.nifi.wali.SnapshotRecovery;
import org.apache.nifi.wali.WriteAheadJournal;
import org.apache.nifi.wali.WriteAheadSnapshot;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.wali.SerDeFactory;
import org.wali.SyncListener;
import org.wali.WriteAheadRepository;

public class SequentialAccessWriteAheadLog<T>
implements WriteAheadRepository<T> {
    private static final int PARTITION_INDEX = 0;
    private static final Logger logger = LoggerFactory.getLogger(SequentialAccessWriteAheadLog.class);
    private static final Pattern JOURNAL_FILENAME_PATTERN = Pattern.compile("\\d+\\.journal");
    private static final int MAX_BUFFERS = 64;
    private static final int BUFFER_SIZE = 262144;
    private final File storageDirectory;
    private final File journalsDirectory;
    protected final SerDeFactory<T> serdeFactory;
    private final SyncListener syncListener;
    private final Set<String> recoveredSwapLocations = new HashSet<String>();
    private final ReadWriteLock journalRWLock = new ReentrantReadWriteLock();
    private final Lock journalReadLock = this.journalRWLock.readLock();
    private final Lock journalWriteLock = this.journalRWLock.writeLock();
    private final ObjectPool<ByteArrayDataOutputStream> streamPool = new BlockingQueuePool<ByteArrayDataOutputStream>(64, () -> new ByteArrayDataOutputStream(262144), stream -> stream.getByteArrayOutputStream().size() < 262144, stream -> stream.getByteArrayOutputStream().reset());
    private final WriteAheadSnapshot<T> snapshot;
    private final RecordLookup<T> recordLookup;
    private SnapshotRecovery<T> snapshotRecovery;
    private volatile boolean recovered = false;
    private WriteAheadJournal<T> journal;
    private volatile long nextTransactionId = 0L;

    public SequentialAccessWriteAheadLog(File storageDirectory, SerDeFactory<T> serdeFactory) throws IOException {
        this(storageDirectory, serdeFactory, SyncListener.NOP_SYNC_LISTENER);
    }

    public SequentialAccessWriteAheadLog(File storageDirectory, SerDeFactory<T> serdeFactory, SyncListener syncListener) throws IOException {
        if (!storageDirectory.exists() && !storageDirectory.mkdirs()) {
            throw new IOException("Directory " + storageDirectory + " does not exist and cannot be created");
        }
        if (!storageDirectory.isDirectory()) {
            throw new IOException("File " + storageDirectory + " is a regular file and not a directory");
        }
        HashMapSnapshot<T> hashMapSnapshot = new HashMapSnapshot<T>(storageDirectory, serdeFactory);
        this.snapshot = hashMapSnapshot;
        this.recordLookup = hashMapSnapshot;
        this.storageDirectory = storageDirectory;
        this.journalsDirectory = new File(storageDirectory, "journals");
        if (!this.journalsDirectory.exists() && !this.journalsDirectory.mkdirs()) {
            throw new IOException("Directory " + this.journalsDirectory + " does not exist and cannot be created");
        }
        this.recovered = false;
        this.serdeFactory = serdeFactory;
        this.syncListener = syncListener == null ? SyncListener.NOP_SYNC_LISTENER : syncListener;
    }

    @Override
    public int update(Collection<T> records, boolean forceSync) throws IOException {
        if (!this.recovered) {
            throw new IllegalStateException("Cannot update repository until record recovery has been performed");
        }
        this.journalReadLock.lock();
        try {
            this.journal.update(records, this.recordLookup);
            if (forceSync) {
                this.journal.fsync();
                this.syncListener.onSync(0);
            }
            this.snapshot.update(records);
        }
        finally {
            this.journalReadLock.unlock();
        }
        return 0;
    }

    @Override
    public synchronized Collection<T> recoverRecords() throws IOException {
        if (this.recovered) {
            throw new IllegalStateException("Cannot recover records from repository because record recovery has already commenced");
        }
        logger.info("Recovering records from Write-Ahead Log at {}", (Object)this.storageDirectory);
        long recoverStart = System.nanoTime();
        this.recovered = true;
        this.snapshotRecovery = this.snapshot.recover();
        this.recoveredSwapLocations.addAll(this.snapshotRecovery.getRecoveredSwapLocations());
        long snapshotRecoveryMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - recoverStart);
        Map<Object, T> recoveredRecords = this.snapshotRecovery.getRecords();
        Set<String> swapLocations = this.snapshotRecovery.getRecoveredSwapLocations();
        File[] journalFiles = this.journalsDirectory.listFiles(this::isJournalFile);
        if (journalFiles == null) {
            throw new IOException("Cannot access the list of files in directory " + this.journalsDirectory + "; please ensure that appropriate file permissions are set.");
        }
        if (this.snapshotRecovery.getRecoveryFile() == null) {
            logger.info("No Snapshot File to recover from at {}. Now recovering records from {} journal files", (Object)this.storageDirectory, (Object)journalFiles.length);
        } else {
            logger.info("Successfully recovered {} records and {} swap files from Snapshot at {} with Max Transaction ID of {} in {} milliseconds. Now recovering records from {} journal files", new Object[]{recoveredRecords.size(), swapLocations.size(), this.snapshotRecovery.getRecoveryFile(), this.snapshotRecovery.getMaxTransactionId(), snapshotRecoveryMillis, journalFiles.length});
        }
        List<File> orderedJournalFiles = Arrays.asList(journalFiles);
        Collections.sort(orderedJournalFiles, new Comparator<File>(){

            @Override
            public int compare(File o1, File o2) {
                long transactionId1 = SequentialAccessWriteAheadLog.this.getMinTransactionId(o1);
                long transactionId2 = SequentialAccessWriteAheadLog.this.getMinTransactionId(o2);
                return Long.compare(transactionId1, transactionId2);
            }
        });
        long snapshotTransactionId = this.snapshotRecovery.getMaxTransactionId();
        int totalUpdates = 0;
        int journalFilesRecovered = 0;
        int journalFilesSkipped = 0;
        long maxTransactionId = snapshotTransactionId;
        for (File journalFile : orderedJournalFiles) {
            long journalMinTransactionId = this.getMinTransactionId(journalFile);
            if (journalMinTransactionId < snapshotTransactionId) {
                logger.debug("Will not recover records from journal file {} because the minimum Transaction ID for that journal is {} and the Transaction ID recovered from Snapshot was {}", new Object[]{journalFile, journalMinTransactionId, snapshotTransactionId});
                ++journalFilesSkipped;
                continue;
            }
            logger.debug("Min Transaction ID for journal {} is {}, so will recover records from journal", (Object)journalFile, (Object)journalMinTransactionId);
            ++journalFilesRecovered;
            LengthDelimitedJournal<T> journal = new LengthDelimitedJournal<T>(journalFile, this.serdeFactory, this.streamPool, 0L);
            Throwable throwable = null;
            try {
                JournalRecovery journalRecovery = journal.recoverRecords(recoveredRecords, swapLocations);
                int updates = journalRecovery.getUpdateCount();
                logger.debug("Recovered {} updates from journal {}", (Object)updates, (Object)journalFile);
                totalUpdates += updates;
                maxTransactionId = Math.max(maxTransactionId, journalRecovery.getMaxTransactionId());
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (journal == null) continue;
                if (throwable != null) {
                    try {
                        journal.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                journal.close();
            }
        }
        logger.debug("Recovered {} updates from {} journal files and skipped {} journal files because their data was already encapsulated in the snapshot", new Object[]{totalUpdates, journalFilesRecovered, journalFilesSkipped});
        this.nextTransactionId = maxTransactionId + 1L;
        long recoverNanos = System.nanoTime() - recoverStart;
        long recoveryMillis = TimeUnit.MILLISECONDS.convert(recoverNanos, TimeUnit.NANOSECONDS);
        logger.info("Successfully recovered {} records in {} milliseconds. Now checkpointing to ensure that Write-Ahead Log is in a consistent state", (Object)recoveredRecords.size(), (Object)recoveryMillis);
        this.recoveredSwapLocations.addAll(swapLocations);
        this.checkpoint(this.recoveredSwapLocations);
        return recoveredRecords.values();
    }

    private long getMinTransactionId(File journalFile) {
        String filename = journalFile.getName();
        String numeral = filename.substring(0, filename.indexOf("."));
        return Long.parseLong(numeral);
    }

    private boolean isJournalFile(File file) {
        if (!file.isFile()) {
            return false;
        }
        String filename = file.getName();
        return JOURNAL_FILENAME_PATTERN.matcher(filename).matches();
    }

    @Override
    public synchronized Set<String> getRecoveredSwapLocations() throws IOException {
        if (!this.recovered) {
            throw new IllegalStateException("Cannot retrieve the Recovered Swap Locations until record recovery has been performed");
        }
        return Collections.unmodifiableSet(this.recoveredSwapLocations);
    }

    public SnapshotCapture<T> captureSnapshot() {
        return this.snapshot.prepareSnapshot(this.nextTransactionId - 1L);
    }

    @Override
    public int checkpoint() throws IOException {
        return this.checkpoint(null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int checkpoint(Set<String> swapLocations) throws IOException {
        SnapshotCapture<T> snapshotCapture;
        File[] existingJournals;
        long startNanos = System.nanoTime();
        this.journalWriteLock.lock();
        try {
            if (this.journal != null) {
                JournalSummary journalSummary = this.journal.getSummary();
                if (journalSummary.getTransactionCount() == 0 && this.journal.isHealthy()) {
                    logger.debug("Will not checkpoint Write-Ahead Log because no updates have occurred since last checkpoint");
                    int n = this.snapshot.getRecordCount();
                    return n;
                }
                try {
                    this.journal.fsync();
                }
                catch (Exception e) {
                    logger.error("Failed to synch Write-Ahead Log's journal to disk at {}", (Object)this.storageDirectory, (Object)e);
                }
                try {
                    this.journal.close();
                }
                catch (Exception e) {
                    logger.error("Failed to close Journal while attempting to checkpoint Write-Ahead Log at {}", (Object)this.storageDirectory);
                }
                this.nextTransactionId = Math.max(this.nextTransactionId, journalSummary.getLastTransactionId() + 1L);
            }
            this.syncListener.onGlobalSync();
            File[] existingFiles = this.journalsDirectory.listFiles(this::isJournalFile);
            existingJournals = existingFiles == null ? new File[]{} : existingFiles;
            snapshotCapture = swapLocations == null ? this.snapshot.prepareSnapshot(this.nextTransactionId - 1L) : this.snapshot.prepareSnapshot(this.nextTransactionId - 1L, swapLocations);
            File journalFile = new File(this.journalsDirectory, this.nextTransactionId + ".journal");
            while (journalFile.exists()) {
                ++this.nextTransactionId;
                journalFile = new File(this.journalsDirectory, this.nextTransactionId + ".journal");
            }
            this.journal = new LengthDelimitedJournal<T>(journalFile, this.serdeFactory, this.streamPool, this.nextTransactionId);
            this.journal.writeHeader();
            logger.debug("Created new Journal starting with Transaction ID {}", (Object)this.nextTransactionId);
        }
        finally {
            this.journalWriteLock.unlock();
        }
        long stopTheWorldMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
        this.snapshot.writeSnapshot(snapshotCapture);
        for (File existingJournal : existingJournals) {
            LengthDelimitedJournal<T> journal = new LengthDelimitedJournal<T>(existingJournal, this.serdeFactory, this.streamPool, this.nextTransactionId);
            journal.dispose();
        }
        long totalNanos = System.nanoTime() - startNanos;
        long millis = TimeUnit.NANOSECONDS.toMillis(totalNanos);
        logger.info("Checkpointed Write-Ahead Log with {} Records and {} Swap Files in {} milliseconds (Stop-the-world time = {} milliseconds), max Transaction ID {}", new Object[]{snapshotCapture.getRecords().size(), snapshotCapture.getSwapLocations().size(), millis, stopTheWorldMillis, snapshotCapture.getMaxTransactionId()});
        return snapshotCapture.getRecords().size();
    }

    @Override
    public void shutdown() throws IOException {
        this.journalWriteLock.lock();
        try {
            if (this.journal != null) {
                this.journal.close();
            }
        }
        finally {
            this.journalWriteLock.unlock();
        }
    }
}

