/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra.frameworkAdapter.jsf;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.myfaces.orchestra.lib._UrlMatcher;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import java.io.IOException;

/**
 * Configures the JsfFrameworkAdapter.
 * <p>
 * Orchestra accesses information about the request, response, session, etc via a
 * FrameworkAdapter so that it can be used with multiple web tier frameworks. This
 * class selects and configures the JSF version of this adapter.
 * <p>
 * Note that the conversation.jsf.OrchestraServletFilter class executes this class
 * as a "subfilter", so defining this filter is not required if that filter is
 * already defined.
 * <p>
 * If filter config parameter "conversationMessagerClass" is set, then this is 
 * passed to the BasicFrameworkAdapter, meaning that this can be either a 
 * beanname defined in the dependency-injection framework, or an absolute
 * classname of a type implementing ConversationMessager.
 * <p>
 * Note that this class is generally only needed in Orchestra 1.0. In later 
 * releases the OrchestraFacesContextFactory configures the JsfFrameworkAdapter
 * instead, requiring less configuration. However this class can be used if the
 * OrchestraFacesContextFactory approach cannot be used for some reason.
 */
public class JsfFrameworkAdapterFilter implements Filter
{
    private final static String INIT_CONVERSATION_MESSAGER = "conversationMessagerClass";

    private final Log log = LogFactory.getLog(JsfFrameworkAdapterFilter.class);
    private JsfFrameworkAdapter adapter;
    private _UrlMatcher urlMatcher;

    public void init(FilterConfig filterConfig) throws ServletException
    {
        String conversationMessager = filterConfig.getInitParameter(INIT_CONVERSATION_MESSAGER);

        adapter =  new JsfFrameworkAdapter(conversationMessager);
        urlMatcher = new _UrlMatcher(filterConfig);
    }

    public void doFilter(ServletRequest req, ServletResponse rsp, FilterChain filterChain)
    throws IOException, ServletException
    {
        if (!urlMatcher.accept(req))
        {
            filterChain.doFilter(req, rsp);
            return;
        }

        if (log.isDebugEnabled())
        {
            log.debug("doFilter");
        }
        try
        {
            adapter.beginRequest();
            filterChain.doFilter(req, rsp);
        }
        finally
        {
            adapter.endRequest();
        }
    }

    public void destroy()
    {
    }
}
