/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.driver.internal.core.session.throttling;

import com.datastax.oss.driver.api.core.RequestThrottlingException;
import com.datastax.oss.driver.api.core.config.DefaultDriverOption;
import com.datastax.oss.driver.api.core.config.DriverExecutionProfile;
import com.datastax.oss.driver.api.core.context.DriverContext;
import com.datastax.oss.driver.api.core.session.throttling.RequestThrottler;
import com.datastax.oss.driver.api.core.session.throttling.Throttled;
import com.datastax.oss.driver.shaded.guava.common.annotations.VisibleForTesting;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.concurrent.locks.ReentrantLock;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class ConcurrencyLimitingRequestThrottler
implements RequestThrottler {
    private static final Logger LOG = LoggerFactory.getLogger(ConcurrencyLimitingRequestThrottler.class);
    private final String logPrefix;
    private final int maxConcurrentRequests;
    private final int maxQueueSize;
    private final ReentrantLock lock = new ReentrantLock();
    @GuardedBy(value="lock")
    private int concurrentRequests;
    @GuardedBy(value="lock")
    private final Deque<Throttled> queue = new ArrayDeque<Throttled>();
    @GuardedBy(value="lock")
    private boolean closed;

    public ConcurrencyLimitingRequestThrottler(DriverContext context) {
        this.logPrefix = context.getSessionName();
        DriverExecutionProfile config = context.getConfig().getDefaultProfile();
        this.maxConcurrentRequests = config.getInt(DefaultDriverOption.REQUEST_THROTTLER_MAX_CONCURRENT_REQUESTS);
        this.maxQueueSize = config.getInt(DefaultDriverOption.REQUEST_THROTTLER_MAX_QUEUE_SIZE);
        LOG.debug("[{}] Initializing with maxConcurrentRequests = {}, maxQueueSize = {}", new Object[]{this.logPrefix, this.maxConcurrentRequests, this.maxQueueSize});
    }

    @Override
    public void register(@NonNull Throttled request) {
        this.lock.lock();
        try {
            if (this.closed) {
                LOG.trace("[{}] Rejecting request after shutdown", (Object)this.logPrefix);
                ConcurrencyLimitingRequestThrottler.fail(request, "The session is shutting down");
            } else if (this.queue.isEmpty() && this.concurrentRequests < this.maxConcurrentRequests) {
                LOG.trace("[{}] Starting newly registered request", (Object)this.logPrefix);
                ++this.concurrentRequests;
                request.onThrottleReady(false);
            } else if (this.queue.size() < this.maxQueueSize) {
                LOG.trace("[{}] Enqueuing request", (Object)this.logPrefix);
                this.queue.add(request);
            } else {
                LOG.trace("[{}] Rejecting request because of full queue", (Object)this.logPrefix);
                ConcurrencyLimitingRequestThrottler.fail(request, String.format("The session has reached its maximum capacity (concurrent requests: %d, queue size: %d)", this.maxConcurrentRequests, this.maxQueueSize));
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void signalSuccess(@NonNull Throttled request) {
        this.lock.lock();
        try {
            this.onRequestDone();
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void signalError(@NonNull Throttled request, @NonNull Throwable error) {
        this.signalSuccess(request);
    }

    @Override
    public void signalTimeout(@NonNull Throttled request) {
        this.lock.lock();
        try {
            if (!this.closed) {
                if (this.queue.remove(request)) {
                    LOG.trace("[{}] Removing timed out request from the queue", (Object)this.logPrefix);
                } else {
                    this.onRequestDone();
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void onRequestDone() {
        assert (this.lock.isHeldByCurrentThread());
        if (!this.closed) {
            if (this.queue.isEmpty()) {
                --this.concurrentRequests;
            } else {
                LOG.trace("[{}] Starting dequeued request", (Object)this.logPrefix);
                this.queue.poll().onThrottleReady(true);
            }
        }
    }

    @Override
    public void close() {
        this.lock.lock();
        try {
            this.closed = true;
            LOG.debug("[{}] Rejecting {} queued requests after shutdown", (Object)this.logPrefix, (Object)this.queue.size());
            for (Throttled request : this.queue) {
                ConcurrencyLimitingRequestThrottler.fail(request, "The session is shutting down");
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public int getQueueSize() {
        this.lock.lock();
        try {
            int n = this.queue.size();
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @VisibleForTesting
    int getConcurrentRequests() {
        this.lock.lock();
        try {
            int n = this.concurrentRequests;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @VisibleForTesting
    Deque<Throttled> getQueue() {
        this.lock.lock();
        try {
            Deque<Throttled> deque = this.queue;
            return deque;
        }
        finally {
            this.lock.unlock();
        }
    }

    private static void fail(Throttled request, String message) {
        request.onThrottleFailure(new RequestThrottlingException(message));
    }
}

