/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.feed.jmx;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import groovy.time.TimeDuration;
import java.io.IOException;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import javax.management.Attribute;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.JMX;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.TabularData;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;
import javax.management.remote.jmxmp.JMXMPConnector;
import javax.management.remote.rmi.RMIConnector;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import org.apache.brooklyn.api.entity.Entity;
import org.apache.brooklyn.entity.java.JmxSupport;
import org.apache.brooklyn.entity.java.UsesJmx;
import org.apache.brooklyn.util.JavaGroovyEquivalents;
import org.apache.brooklyn.util.collections.MutableMap;
import org.apache.brooklyn.util.core.crypto.SecureKeys;
import org.apache.brooklyn.util.crypto.SslTrustUtils;
import org.apache.brooklyn.util.exceptions.Exceptions;
import org.apache.brooklyn.util.exceptions.RuntimeInterruptedException;
import org.apache.brooklyn.util.repeat.Repeater;
import org.apache.brooklyn.util.time.Duration;
import org.apache.brooklyn.util.time.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JmxHelper {
    private static final Logger LOG = LoggerFactory.getLogger(JmxHelper.class);
    public static final String JMX_URL_FORMAT = "service:jmx:rmi:///jndi/rmi://%s:%d/%s";
    public static final String RMI_JMX_URL_FORMAT = "service:jmx:rmi://%s:%d/jndi/rmi://%s:%d/%s";
    public static final String JMXMP_URL_FORMAT = "service:jmx:jmxmp://%s:%d";
    private static final Map<String, Set<ObjectName>> notFoundMBeansByUrl = Collections.synchronizedMap(new WeakHashMap());
    public static final Map<String, String> CLASSES = ImmutableMap.builder().put((Object)"Integer", (Object)Integer.TYPE.getName()).put((Object)"Long", (Object)Long.TYPE.getName()).put((Object)"Boolean", (Object)Boolean.TYPE.getName()).put((Object)"Byte", (Object)Byte.TYPE.getName()).put((Object)"Character", (Object)Character.TYPE.getName()).put((Object)"Double", (Object)Double.TYPE.getName()).put((Object)"Float", (Object)Float.TYPE.getName()).put((Object)"GStringImpl", (Object)String.class.getName()).put((Object)"LinkedHashMap", (Object)Map.class.getName()).put((Object)"TreeMap", (Object)Map.class.getName()).put((Object)"HashMap", (Object)Map.class.getName()).put((Object)"ConcurrentHashMap", (Object)Map.class.getName()).put((Object)"TabularDataSupport", (Object)TabularData.class.getName()).put((Object)"CompositeDataSupport", (Object)CompositeData.class.getName()).build();
    final Entity entity;
    final String url;
    final String user;
    final String password;
    private volatile transient JMXConnector connector;
    private volatile transient MBeanServerConnection connection;
    private transient boolean triedConnecting;
    private transient boolean failedReconnecting;
    private transient long failedReconnectingTime;
    private int minTimeBetweenReconnectAttempts = 1000;
    private final AtomicBoolean terminated = new AtomicBoolean();
    private final Set<ObjectName> notFoundMBeans;

    public static String toJmxUrl(Entity entity) {
        String url = (String)entity.getAttribute(UsesJmx.JMX_URL);
        if (url != null) {
            return url;
        }
        new JmxSupport(entity, null).setJmxUrl();
        url = (String)entity.getAttribute(UsesJmx.JMX_URL);
        return (String)Preconditions.checkNotNull((Object)url, (Object)("Could not find URL for " + entity));
    }

    public static String toRmiJmxUrl(String host, Integer jmxRmiServerPort, Integer rmiRegistryPort, String context) {
        if (rmiRegistryPort != null && rmiRegistryPort > 0) {
            if (jmxRmiServerPort != null && jmxRmiServerPort > 0 && jmxRmiServerPort != rmiRegistryPort) {
                return String.format(RMI_JMX_URL_FORMAT, host, jmxRmiServerPort, host, rmiRegistryPort, context);
            }
            return String.format(JMX_URL_FORMAT, host, rmiRegistryPort, context);
        }
        if (jmxRmiServerPort != null && jmxRmiServerPort > 0) {
            LOG.warn("No RMI registry port set for " + host + "; attempting to use JMX port for RMI lookup");
            return String.format(JMX_URL_FORMAT, host, jmxRmiServerPort, context);
        }
        LOG.warn("No RMI/JMX details set for " + host + "; returning null");
        return null;
    }

    public static String toJmxmpUrl(String host, Integer jmxmpPort) {
        return "service:jmx:jmxmp://" + host + (jmxmpPort != null ? ":" + jmxmpPort : "");
    }

    public JmxHelper(Entity entity) {
        this(JmxHelper.toJmxUrl(entity), entity, (String)entity.getAttribute(UsesJmx.JMX_USER), (String)entity.getAttribute(UsesJmx.JMX_PASSWORD));
        if (entity.getAttribute(UsesJmx.JMX_URL) == null) {
            entity.sensors().set(UsesJmx.JMX_URL, (Object)this.url);
        }
    }

    public JmxHelper(String url) {
        this(url, null, null);
    }

    public JmxHelper(String url, String user, String password) {
        this(url, null, user, password);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public JmxHelper(String url, Entity entity, String user, String password) {
        this.url = url;
        this.entity = entity;
        this.user = user;
        this.password = password;
        Map<String, Set<ObjectName>> map = notFoundMBeansByUrl;
        synchronized (map) {
            Set<ObjectName> set = notFoundMBeansByUrl.get(url);
            if (set == null) {
                set = Collections.synchronizedSet(Collections.newSetFromMap(new WeakHashMap()));
                notFoundMBeansByUrl.put(url, set);
            }
            this.notFoundMBeans = set;
        }
    }

    public void setMinTimeBetweenReconnectAttempts(int val) {
        this.minTimeBetweenReconnectAttempts = val;
    }

    public String getUrl() {
        return this.url;
    }

    protected MBeanServerConnection getConnection() {
        return this.connection;
    }

    public boolean isConnected() {
        return this.connection != null;
    }

    public synchronized void reconnectWithRetryDampened() throws IOException {
        long timeSince;
        if (this.failedReconnecting && (timeSince = System.currentTimeMillis() - this.failedReconnectingTime) < (long)this.minTimeBetweenReconnectAttempts) {
            String msg = "Not reconnecting to JMX at " + this.url + " because attempt failed " + Time.makeTimeStringRounded((long)timeSince) + " ago";
            throw new IllegalStateException(msg);
        }
        this.reconnect();
    }

    public synchronized void reconnect() throws IOException {
        this.disconnect();
        try {
            this.connect();
            this.failedReconnecting = false;
        }
        catch (Exception e) {
            if (this.failedReconnecting) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("unable to re-connect to JMX url (repeated failure): {}: {}", (Object)this.url, (Object)e);
                }
            } else {
                LOG.debug("unable to re-connect to JMX url {} (rethrowing): {}", (Object)this.url, (Object)e);
                this.failedReconnecting = true;
            }
            this.failedReconnectingTime = System.currentTimeMillis();
            throw Throwables.propagate((Throwable)e);
        }
    }

    public synchronized void connect() throws IOException {
        if (this.terminated.get()) {
            throw new IllegalStateException("JMX Helper " + this + " already terminated");
        }
        if (this.connection != null) {
            return;
        }
        this.triedConnecting = true;
        if (this.connector != null) {
            this.connector.close();
        }
        JMXServiceURL serviceUrl = new JMXServiceURL(this.url);
        Map env = this.getConnectionEnvVars();
        try {
            this.connector = JmxHelper.newConnector(serviceUrl, env);
        }
        catch (NullPointerException npe) {
            boolean thrownByWso2 = npe.getStackTrace()[0].toString().contains("org.wso2.carbon.core.security.CarbonJMXAuthenticator.authenticate");
            if (thrownByWso2) {
                throw new IOException("Failed to connect to url " + this.url + ". NullPointerException is thrown, but replaced by an IOException to fix a WSO2 JMX problem", npe);
            }
            throw npe;
        }
        catch (IOException e) {
            Exceptions.propagateIfFatal((Throwable)e);
            if (this.terminated.get()) {
                throw new IllegalStateException("JMX Helper " + this + " already terminated", e);
            }
            throw e;
        }
        this.connection = this.connector.getMBeanServerConnection();
        if (this.terminated.get()) {
            this.disconnectNow();
            throw new IllegalStateException("JMX Helper " + this + " already terminated");
        }
    }

    public static JMXConnector newConnector(JMXServiceURL url, Map<String, ?> env) throws IOException {
        MutableMap envCopy = MutableMap.copyOf(env);
        String protocol = url.getProtocol();
        if ("jmxmp".equalsIgnoreCase(protocol)) {
            envCopy.put("jmx.remote.protocol.provider.class.loader", JMXMPConnector.class.getClassLoader());
            envCopy.put("jmx.remote.default.class.loader", JMXMPConnector.class.getClassLoader());
        } else if ("rmi".equalsIgnoreCase(protocol)) {
            envCopy.put("jmx.remote.protocol.provider.class.loader", RMIConnector.class.getClassLoader());
            envCopy.put("jmx.remote.default.class.loader", RMIConnector.class.getClassLoader());
        }
        return JMXConnectorFactory.connect(url, envCopy);
    }

    public Map getConnectionEnvVars() {
        LinkedHashMap<String, Object> env = new LinkedHashMap<String, Object>();
        if (JavaGroovyEquivalents.groovyTruth((String)this.user) && JavaGroovyEquivalents.groovyTruth((String)this.password)) {
            String[] creds = new String[]{this.user, this.password};
            env.put("jmx.remote.credentials", creds);
        }
        if (this.entity != null && JavaGroovyEquivalents.groovyTruth((Object)this.entity.getConfig(UsesJmx.JMX_SSL_ENABLED))) {
            env.put("jmx.remote.profiles", "TLS");
            PrivateKey key = (PrivateKey)this.entity.getConfig(UsesJmx.JMX_SSL_ACCESS_KEY);
            Certificate cert = (Certificate)this.entity.getConfig(UsesJmx.JMX_SSL_ACCESS_CERT);
            KeyStore ks = SecureKeys.newKeyStore();
            try {
                KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
                if (key != null) {
                    ks.setKeyEntry("brooklyn-jmx-access", key, "".toCharArray(), new Certificate[]{cert});
                }
                kmf.init(ks, "".toCharArray());
                TrustManager tms = SslTrustUtils.TRUST_ALL;
                SSLContext ctx = SSLContext.getInstance("TLSv1");
                ctx.init(kmf.getKeyManagers(), new TrustManager[]{tms}, null);
                SSLSocketFactory ssf = ctx.getSocketFactory();
                env.put("jmx.remote.tls.socket.factory", ssf);
            }
            catch (Exception e) {
                LOG.warn("Error setting key " + key + " for " + this.entity + ": " + e, (Throwable)e);
            }
        }
        return env;
    }

    public boolean connect(long timeoutMs) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connecting to JMX URL: {} ({})", (Object)this.url, (Object)(timeoutMs == -1L ? "indefinitely" : timeoutMs + "ms timeout"));
        }
        long startMs = System.currentTimeMillis();
        long endMs = timeoutMs == -1L ? Long.MAX_VALUE : startMs + timeoutMs;
        long currentTime = startMs;
        Exception lastError = null;
        int attempt = 0;
        while (currentTime <= endMs) {
            currentTime = System.currentTimeMillis();
            if (attempt != 0) {
                JmxHelper.sleep(100L);
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace("trying connection to {} at time {}", (Object)this.url, (Object)currentTime);
            }
            try {
                this.connect();
                return true;
            }
            catch (Exception e) {
                Exceptions.propagateIfFatal((Throwable)e);
                if (!this.terminated.get() && this.shouldRetryOn(e)) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Attempt {} failed connecting to {} ({})", new Object[]{attempt + 1, this.url, e.getMessage()});
                    }
                } else {
                    throw Exceptions.propagate((Throwable)e);
                }
                lastError = e;
                ++attempt;
            }
        }
        LOG.warn("unable to connect to JMX url: " + this.url, lastError);
        return false;
    }

    private boolean shouldRetryOn(Exception e) {
        if (e instanceof AttributeNotFoundException) {
            return false;
        }
        if (e instanceof InstanceAlreadyExistsException) {
            return false;
        }
        if (e instanceof InstanceNotFoundException) {
            return false;
        }
        if (e instanceof InvalidAttributeValueException) {
            return false;
        }
        if (e instanceof ListenerNotFoundException) {
            return false;
        }
        if (e instanceof MalformedObjectNameException) {
            return false;
        }
        if (e instanceof NotCompliantMBeanException) {
            return false;
        }
        if (e instanceof InterruptedException) {
            return false;
        }
        return !(e instanceof RuntimeInterruptedException);
    }

    public synchronized void disconnect() {
        this.disconnectNow();
    }

    public void terminate() {
        this.terminated.set(true);
        this.disconnectNow();
    }

    protected void disconnectNow() {
        this.triedConnecting = false;
        if (this.connector != null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Disconnecting from JMX URL {}", (Object)this.url);
            }
            try {
                this.connector.close();
            }
            catch (Exception e) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Caught exception disconnecting from JMX at {} ({})", (Object)this.url, (Object)e.getMessage());
                }
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Details for exception disconnecting JMX", (Throwable)e);
                }
            }
            finally {
                this.connector = null;
                this.connection = null;
            }
        }
    }

    private synchronized MBeanServerConnection getConnectionOrFail() {
        if (this.isConnected()) {
            return this.getConnection();
        }
        if (this.triedConnecting) {
            throw new IllegalStateException("Failed to connect to JMX at " + this.url);
        }
        String msg = "Not connected (and not attempted to connect) to JMX at " + this.url + (this.failedReconnecting ? " (last reconnect failure at " + Time.makeDateString((long)this.failedReconnectingTime) + ")" : "");
        throw new IllegalStateException(msg);
    }

    private <T> T invokeWithReconnect(Callable<T> task) {
        try {
            return task.call();
        }
        catch (Exception e) {
            if (this.shouldRetryOn(e)) {
                try {
                    this.reconnectWithRetryDampened();
                    return task.call();
                }
                catch (Exception e2) {
                    throw Throwables.propagate((Throwable)e2);
                }
            }
            throw Throwables.propagate((Throwable)e);
        }
    }

    public ObjectName toLiteralObjectName(ObjectName objectName) {
        if (((ObjectName)Preconditions.checkNotNull((Object)objectName, (Object)"objectName")).isPattern()) {
            ObjectInstance bean = this.findMBean(objectName);
            return bean != null ? bean.getObjectName() : null;
        }
        return objectName;
    }

    public Set<ObjectInstance> findMBeans(final ObjectName objectName) {
        return this.invokeWithReconnect(new Callable<Set<ObjectInstance>>(){

            @Override
            public Set<ObjectInstance> call() throws Exception {
                return JmxHelper.this.getConnectionOrFail().queryMBeans(objectName, null);
            }
        });
    }

    public ObjectInstance findMBean(ObjectName objectName) {
        Set<ObjectInstance> beans = this.findMBeans(objectName);
        if (beans.size() == 1) {
            this.notFoundMBeans.remove(objectName);
            return (ObjectInstance)Iterables.getOnlyElement(beans);
        }
        boolean changed = this.notFoundMBeans.add(objectName);
        if (beans.size() > 1) {
            if (changed) {
                LOG.warn("JMX object name query returned {} values for {} at {}; ignoring all", new Object[]{beans.size(), objectName.getCanonicalName(), this.url});
            } else if (LOG.isDebugEnabled()) {
                LOG.debug("JMX object name query returned {} values for {} at {} (repeating); ignoring all", new Object[]{beans.size(), objectName.getCanonicalName(), this.url});
            }
        } else if (changed) {
            LOG.warn("JMX object {} not found at {}", (Object)objectName.getCanonicalName(), (Object)this.url);
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("JMX object {} not found at {} (repeating)", (Object)objectName.getCanonicalName(), (Object)this.url);
        }
        return null;
    }

    public Set<ObjectInstance> doesMBeanExistsEventually(ObjectName objectName, Duration timeout) {
        return this.doesMBeanExistsEventually(objectName, timeout.toMilliseconds(), TimeUnit.MILLISECONDS);
    }

    @Deprecated
    public Set<ObjectInstance> doesMBeanExistsEventually(ObjectName objectName, TimeDuration timeout) {
        return this.doesMBeanExistsEventually(objectName, timeout.toMilliseconds(), TimeUnit.MILLISECONDS);
    }

    public Set<ObjectInstance> doesMBeanExistsEventually(ObjectName objectName, long timeoutMillis) {
        return this.doesMBeanExistsEventually(objectName, timeoutMillis, TimeUnit.MILLISECONDS);
    }

    public Set<ObjectInstance> doesMBeanExistsEventually(String objectName, Duration timeout) {
        return this.doesMBeanExistsEventually(JmxHelper.createObjectName(objectName), timeout);
    }

    @Deprecated
    public Set<ObjectInstance> doesMBeanExistsEventually(String objectName, TimeDuration timeout) {
        return this.doesMBeanExistsEventually(JmxHelper.createObjectName(objectName), timeout);
    }

    public Set<ObjectInstance> doesMBeanExistsEventually(String objectName, long timeout, TimeUnit timeUnit) {
        return this.doesMBeanExistsEventually(JmxHelper.createObjectName(objectName), timeout, timeUnit);
    }

    public Set<ObjectInstance> doesMBeanExistsEventually(final ObjectName objectName, long timeout, TimeUnit timeUnit) {
        final long timeoutMillis = timeUnit.toMillis(timeout);
        final AtomicReference<ImmutableSet> beans = new AtomicReference<ImmutableSet>(ImmutableSet.of());
        try {
            Repeater.create((String)("Wait for " + objectName)).limitTimeTo(timeout, timeUnit).every(500L, TimeUnit.MILLISECONDS).until((Callable)new Callable<Boolean>(){

                @Override
                public Boolean call() {
                    JmxHelper.this.connect(timeoutMillis);
                    beans.set(JmxHelper.this.findMBeans(objectName));
                    return !((Set)beans.get()).isEmpty();
                }
            }).rethrowException().run();
            return (Set)beans.get();
        }
        catch (Exception e) {
            throw Exceptions.propagate((Throwable)e);
        }
    }

    public void assertMBeanExistsEventually(ObjectName objectName, Duration timeout) {
        this.assertMBeanExistsEventually(objectName, timeout.toMilliseconds(), TimeUnit.MILLISECONDS);
    }

    @Deprecated
    public void assertMBeanExistsEventually(ObjectName objectName, TimeDuration timeout) {
        this.assertMBeanExistsEventually(objectName, timeout.toMilliseconds(), TimeUnit.MILLISECONDS);
    }

    public void assertMBeanExistsEventually(ObjectName objectName, long timeoutMillis) {
        this.assertMBeanExistsEventually(objectName, timeoutMillis, TimeUnit.MILLISECONDS);
    }

    public void assertMBeanExistsEventually(ObjectName objectName, long timeout, TimeUnit timeUnit) {
        Set<ObjectInstance> beans = this.doesMBeanExistsEventually(objectName, timeout, timeUnit);
        if (beans.size() != 1) {
            throw new IllegalStateException("MBean " + objectName + " not found within " + timeout + (beans.size() > 1 ? "; found multiple matches: " + beans : ""));
        }
    }

    public Object getAttribute(ObjectName objectName, final String attribute) {
        final ObjectName realObjectName = this.toLiteralObjectName(objectName);
        if (realObjectName != null) {
            Object result = this.invokeWithReconnect(new Callable<Object>(){

                @Override
                public Object call() throws Exception {
                    return JmxHelper.this.getConnectionOrFail().getAttribute(realObjectName, attribute);
                }
            });
            if (LOG.isTraceEnabled()) {
                LOG.trace("From {}, for jmx attribute {}.{}, got value {}", new Object[]{this.url, objectName.getCanonicalName(), attribute, result});
            }
            return result;
        }
        return null;
    }

    public void setAttribute(String objectName, String attribute, Object val) {
        this.setAttribute(JmxHelper.createObjectName(objectName), attribute, val);
    }

    public void setAttribute(ObjectName objectName, final String attribute, final Object val) {
        final ObjectName realObjectName = this.toLiteralObjectName(objectName);
        if (realObjectName != null) {
            this.invokeWithReconnect(new Callable<Void>(){

                @Override
                public Void call() throws Exception {
                    JmxHelper.this.getConnectionOrFail().setAttribute(realObjectName, new Attribute(attribute, val));
                    return null;
                }
            });
            if (LOG.isTraceEnabled()) {
                LOG.trace("From {}, for jmx attribute {}.{}, set value {}", new Object[]{this.url, objectName.getCanonicalName(), attribute, val});
            }
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("From {}, cannot set attribute {}.{}, because mbean not found", new Object[]{this.url, objectName.getCanonicalName(), attribute});
        }
    }

    public Object operation(String objectName, String method, Object ... arguments) {
        return this.operation(JmxHelper.createObjectName(objectName), method, arguments);
    }

    public Object operation(ObjectName objectName, final String method, final Object ... arguments) {
        final ObjectName realObjectName = this.toLiteralObjectName(objectName);
        final String[] signature = new String[arguments.length];
        for (int i = 0; i < arguments.length; ++i) {
            Class<?> clazz = arguments[i].getClass();
            signature[i] = CLASSES.containsKey(clazz.getSimpleName()) ? CLASSES.get(clazz.getSimpleName()) : clazz.getName();
        }
        Object result = this.invokeWithReconnect(new Callable<Object>(){

            @Override
            public Object call() throws Exception {
                return JmxHelper.this.getConnectionOrFail().invoke(realObjectName, method, arguments, signature);
            }
        });
        if (LOG.isTraceEnabled()) {
            LOG.trace("From {}, for jmx operation {}.{}({}), got value {}", new Object[]{this.url, realObjectName.getCanonicalName(), method, Arrays.asList(arguments), result});
        }
        return result;
    }

    public void addNotificationListener(String objectName, NotificationListener listener) {
        this.addNotificationListener(JmxHelper.createObjectName(objectName), listener, null);
    }

    public void addNotificationListener(String objectName, NotificationListener listener, NotificationFilter filter) {
        this.addNotificationListener(JmxHelper.createObjectName(objectName), listener, filter);
    }

    public void addNotificationListener(ObjectName objectName, NotificationListener listener) {
        this.addNotificationListener(objectName, listener, null);
    }

    public void addNotificationListener(final ObjectName objectName, final NotificationListener listener, final NotificationFilter filter) {
        this.invokeWithReconnect(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                JmxHelper.this.getConnectionOrFail().addNotificationListener(objectName, listener, filter, null);
                return null;
            }
        });
    }

    public void removeNotificationListener(String objectName, NotificationListener listener) {
        this.removeNotificationListener(JmxHelper.createObjectName(objectName), listener);
    }

    public void removeNotificationListener(ObjectName objectName, NotificationListener listener) {
        this.removeNotificationListener(objectName, listener, null);
    }

    public void removeNotificationListener(final ObjectName objectName, final NotificationListener listener, final NotificationFilter filter) {
        if (this.isConnected()) {
            this.invokeWithReconnect(new Callable<Void>(){

                @Override
                public Void call() throws Exception {
                    JmxHelper.this.getConnectionOrFail().removeNotificationListener(objectName, listener, filter, null);
                    return null;
                }
            });
        }
    }

    public <M> M getProxyObject(String objectName, Class<M> mbeanInterface) {
        return this.getProxyObject(JmxHelper.createObjectName(objectName), mbeanInterface);
    }

    public <M> M getProxyObject(ObjectName objectName, Class<M> mbeanInterface) {
        MBeanServerConnection connection = this.getConnectionOrFail();
        return JMX.newMBeanProxy(connection, objectName, mbeanInterface, false);
    }

    public static ObjectName createObjectName(String name) {
        try {
            return new ObjectName(name);
        }
        catch (MalformedObjectNameException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private static void sleep(long sleepTimeMillis) {
        try {
            Thread.sleep(sleepTimeMillis);
        }
        catch (InterruptedException e) {
            throw new RuntimeInterruptedException(e);
        }
    }
}

